// =============================================
// popup.js - Unicode Text Styler Core Logic
// Now with Suggestion Feature
// =============================================

// Unicode styles configuration
const unicodeStyles = [
  { id: "bold", name: "Bold" },
  { id: "italic", name: "Italic" },
  { id: "script", name: "Script" },
  { id: "monospace", name: "Monospace" },
  { id: "fraktur", name: "Fraktur" },
  { id: "doubleStruck", name: "Double Struck" },
  { id: "strikethrough", name: "Strikethrough" },
];

const MAX_HISTORY = 50;
let historyStack = [];
let futureStack = [];
let isApplyingStyle = false;
let reverseStyleMap = null; // Added for multi-style support

/**
 * Strips HTML tags from a string
 * @param {string} html - Input string with potential HTML
 * @returns {string} Plain text string
 */
function stripHtml(html) {
  let tmp = document.createElement("DIV");
  tmp.innerHTML = html;
  return tmp.textContent || tmp.innerText || "";
}

/**
 * Converts text to styled Unicode characters
 * @param {string} text - Input text to convert
 * @param {string} style - Style key (bold/italic/script/monospace)
 * @returns {string} Styled text
 */
function convertText(text, style) {
  const { styleMaps } = window.UnicodeStyler;
  if (!text || !styleMaps[style]) return text;

  return text
    .split("")
    .map((char) => {
      // 1. Try exact character match
      if (styleMaps[style][char]) return styleMaps[style][char];

      // 2. Try opposite case
      const oppositeCase =
        char === char.toLowerCase() ? char.toUpperCase() : char.toLowerCase();

      return styleMaps[style][oppositeCase] || char;
    })
    .join("");
}

// New helper functions for multi-style support
function ensureReverseMap() {
  // If reverseStyleMap was successfully built previously and has content, no need to rebuild.
  if (reverseStyleMap && Object.keys(reverseStyleMap).length > 0) {
    return;
  }

  // Check if styleMaps is available and has content.
  if (
    !window.UnicodeStyler ||
    !window.UnicodeStyler.styleMaps ||
    Object.keys(window.UnicodeStyler.styleMaps).length === 0
  ) {
    // console.warn("UnicodeStyler.styleMaps not available or empty for reverse map generation. Will attempt to build later.");
    reverseStyleMap = {}; // Initialize to empty if maps aren't ready
    return;
  }

  const localReverseMap = {}; // Build into a temporary map first
  const { styleMaps } = window.UnicodeStyler;

  const stylePriority = ["boldItalic"]; // Prioritize combined styles
  const allStyleNames = Object.keys(styleMaps);
  const sortedStyleNames = [
    ...stylePriority.filter((s) => allStyleNames.includes(s)),
    ...allStyleNames.filter((s) => !stylePriority.includes(s) && styleMaps[s]), // Ensure map exists for the style
  ];

  const basicLatinLetterOrDigit = /^[a-zA-Z0-9]$/;

  for (const styleName of sortedStyleNames) {
    const map = styleMaps[styleName];
    if (!map) continue;
    for (const plainChar in map) {
      if (Object.hasOwnProperty.call(map, plainChar)) {
        const styledChar = map[plainChar];

        // If the styledChar is a basic letter/digit AND it's different from the plainChar it's derived from
        // (e.g., 'u' derived from 'n' in upsideDown map), then we don't want to add this to the reverseMap
        // because it makes the basic character 'u' behave as if it's a styled 'n'.
        if (
          basicLatinLetterOrDigit.test(styledChar) &&
          styledChar !== plainChar
        ) {
          // Skip this mapping for the reverse map to avoid ambiguity with plain characters.
        } else {
          // Original logic for adding to reverse map based on priority from sortedStyleNames
          // If styledChar is not already in localReverseMap, add it.
          // (Priority is handled by the order of sortedStyleNames: first one wins)
          if (!localReverseMap[styledChar]) {
            localReverseMap[styledChar] = {
              plain: plainChar,
              style: styleName,
            };
          }
        }
      }
    }
  }

  // Only assign to the global variable if the build resulted in a non-empty map.
  if (Object.keys(localReverseMap).length > 0) {
    reverseStyleMap = localReverseMap;
    // console.log("Reverse map (re)built successfully:", reverseStyleMap);
  } else {
    // console.warn("Reverse map build resulted in an empty map. StyleMaps might be incomplete or empty.");
    reverseStyleMap = {}; // Keep it as an empty object if build failed to produce entries
  }
}

function getCharPlainAndStyle(char) {
  ensureReverseMap(); // Make sure map is built
  if (reverseStyleMap && reverseStyleMap[char]) {
    return reverseStyleMap[char];
  }
  return { plain: char, style: null }; // Not a known styled character
}

function calculateResultingStyle(existingStyle, appliedStyle) {
  if (existingStyle === appliedStyle) {
    return null; // Toggle off
  }

  if (appliedStyle === "bold") {
    if (existingStyle === "italic") return "boldItalic";
    if (existingStyle === "boldItalic") return "italic";
    return "bold";
  }

  if (appliedStyle === "italic") {
    if (existingStyle === "bold") return "boldItalic";
    if (existingStyle === "boldItalic") return "bold";
    return "italic";
  }

  if (appliedStyle === "boldItalic") {
    return "boldItalic"; // Overrides or applies
  }

  // For other exclusive styles (script, fraktur, etc.)
  return appliedStyle; // New style overrides
}

function convertSingleChar(plainChar, style) {
  if (!style) return plainChar;

  const { styleMaps } = window.UnicodeStyler;
  if (!styleMaps || !styleMaps[style]) {
    console.warn(`Style map for '${style}' not found.`);
    return plainChar;
  }

  const map = styleMaps[style];
  let styledChar = map[plainChar];

  if (styledChar === undefined) {
    // Try opposite case for letters
    if (plainChar.match(/[a-zA-Z]/)) {
      const oppositeCaseChar =
        plainChar === plainChar.toLowerCase()
          ? plainChar.toUpperCase()
          : plainChar.toLowerCase();
      styledChar = map[oppositeCaseChar]; // Correctly assign the result of the opposite case lookup
    }
  }

  return styledChar !== undefined ? styledChar : plainChar;
}

/**
 * Applies style to selected text or inserts sample
 * @param {string} newlyAppliedStyle - Style key to apply
 */
function applyStyle(newlyAppliedStyle) {
  if (!window.UnicodeStyler) {
    console.error("UnicodeStyler not loaded");
    return;
  }
  ensureReverseMap(); // Ensure reverse map is built before use

  const editor = document.getElementById("editor");
  if (!editor) return;

  saveState(); // Save state before applying style
  isApplyingStyle = true;

  const currentText = editor.value;
  const selectionStart = editor.selectionStart;
  const selectionEnd = editor.selectionEnd;

  if (selectionStart === selectionEnd) {
    // No text selected
    isApplyingStyle = false;
    // Optionally, could insert sample text or provide feedback
    // For now, do nothing if no selection, consistent with previous behavior for this part
    updateUndoRedoButtons(); // Still call this in case saveState did something
    editor.focus();
    return;
  }

  const selectedChars = currentText
    .substring(selectionStart, selectionEnd)
    .split("");
  let processedText = "";

  for (const char of selectedChars) {
    const { plain: plainChar, style: existingStyle } =
      getCharPlainAndStyle(char);
    const finalStyle = calculateResultingStyle(
      existingStyle,
      newlyAppliedStyle
    );
    processedText += convertSingleChar(plainChar, finalStyle);
  }

  editor.value =
    currentText.substring(0, selectionStart) +
    processedText +
    currentText.substring(selectionEnd);

  // Restore selection
  editor.setSelectionRange(
    selectionStart,
    selectionStart + processedText.length
  );

  isApplyingStyle = false;
  saveState(); // Save state after applying style
  updateUndoRedoButtons();
  editor.focus(); // Keep focus on the editor
}

// New function to apply list formatting
function applyListFormatting(listType) {
  const editor = document.getElementById("editor");
  if (!editor) return;

  const selectionStart = editor.selectionStart;
  const selectionEnd = editor.selectionEnd;
  const currentText = editor.value;

  if (selectionStart === selectionEnd) {
    showTooltip(
      listType === "ordered" ? "ordered-list-btn" : "unordered-list-btn",
      "Select text to list."
    );
    editor.focus();
    return;
  }

  saveState(); // Save state before applying formatting
  isApplyingStyle = true;

  const selectedText = currentText.substring(selectionStart, selectionEnd);
  const lines = selectedText.split("\n");
  let formattedLines = [];

  if (listType === "ordered") {
    let counter = 1;
    formattedLines = lines.map((line) => {
      if (line.trim() === "") return ""; // Keep empty lines as they are
      return `${counter++}. ${line}`;
    });
  } else if (listType === "unordered") {
    formattedLines = lines.map((line) => {
      if (line.trim() === "") return ""; // Keep empty lines as they are
      return `• ${line}`; // Using • as the bullet point
    });
  }

  const processedText = formattedLines.join("\n");

  editor.value =
    currentText.substring(0, selectionStart) +
    processedText +
    currentText.substring(selectionEnd);

  // Restore selection to cover the new list
  editor.setSelectionRange(
    selectionStart,
    selectionStart + processedText.length
  );

  isApplyingStyle = false;
  saveState(); // Save state after applying formatting
  updateUndoRedoButtons();
  showTooltip(
    listType === "ordered" ? "ordered-list-btn" : "unordered-list-btn",
    "List applied!"
  );
  editor.focus();
}

// Show tooltip feedback
function showTooltip(elementId, message) {
  const btn = document.getElementById(elementId);
  if (!btn) {
    console.warn(`Button with id '${elementId}' not found for tooltip.`);
    return;
  }

  // Remove any existing tooltip for this button to prevent duplicates
  const existingTooltip = btn.querySelector(".dynamic-tooltip");
  if (existingTooltip) {
    existingTooltip.remove();
  }

  const tooltip = document.createElement("div");
  tooltip.className =
    "dynamic-tooltip absolute left-0 -translate-x-full px-2 py-1 bg-zinc-700 text-white rounded-md text-xs whitespace-nowrap z-20"; // Consistent background and added z-index
  tooltip.textContent = message;
  btn.style.position = "relative"; // Ensure button is a positioning context for the absolute tooltip
  btn.appendChild(tooltip);

  setTimeout(() => {
    tooltip.classList.add("opacity-0", "transition-opacity", "duration-300");
    setTimeout(() => tooltip.remove(), 300);
  }, 1500);
}

// Initialize undo/redo functionality
function initUndoRedo() {
  const editor = document.getElementById("editor");
  const undoBtn = document.getElementById("undo-btn");
  const redoBtn = document.getElementById("redo-btn");

  if (!editor) return; // Exit if editor doesn't exist

  // Save initial state
  saveState();

  // Editor event listener for content changes
  editor.addEventListener("input", () => {
    if (!isApplyingStyle) {
      saveState();
    }
    updateUndoRedoButtons();
  });

  // Undo button click handler
  undoBtn.addEventListener("click", () => {
    if (historyStack.length > 1) {
      futureStack.push(historyStack.pop());
      restoreState(historyStack[historyStack.length - 1]);
      updateUndoRedoButtons();
    }
  });

  // Redo button click handler
  redoBtn.addEventListener("click", () => {
    if (futureStack.length > 0) {
      const state = futureStack.pop();
      historyStack.push(state);
      restoreState(state);
      updateUndoRedoButtons();
    }
  });

  // Share button click handler
  const shareBtn = document.getElementById("share-btn");
  if (shareBtn) {
    shareBtn.addEventListener("click", async () => {
      const editor = document.getElementById("editor");
      if (!editor || !editor.value) {
        showTooltip("share-btn", "Nothing to share!");
        return;
      }

      const textToShare = editor.value;

      if (navigator.share) {
        try {
          await navigator.share({
            title: "Styled Text",
            text: textToShare,
          });
          // showTooltip("share-btn", "Shared!"); // Optional: if you want feedback after successful native share
        } catch (error) {
          console.error("Error sharing:", error);
          // Fallback to copy if native share fails or is cancelled by user
          copyToClipboard(textToShare, "share-btn", "Failed to share, copied!");
        }
      } else {
        // Fallback for browsers that don't support Web Share API
        copyToClipboard(textToShare, "share-btn", "Copied to clipboard!");
      }
    });
  }

  // Update button states initially
  updateUndoRedoButtons();
}

function saveState() {
  const editor = document.getElementById("editor");
  if (!editor) return; // Exit if editor doesn't exist
  try {
    const currentState = editor.value; // Changed from innerHTML

    // Don't save if same as last state
    if (
      historyStack.length > 0 &&
      historyStack[historyStack.length - 1] === currentState
    ) {
      return;
    }

    historyStack.push(currentState);
    futureStack = []; // Clear redo stack on new action

    // Limit history size
    if (historyStack.length > MAX_HISTORY) {
      historyStack.shift();
    }
  } catch (error) {
    console.error("Error saving state:", error);
    return;
  }
}

function restoreState(state) {
  const editor = document.getElementById("editor");
  if (!editor) return; // Exit if editor doesn't exist
  isApplyingStyle = true;
  editor.value = state; // Changed from innerHTML
  isApplyingStyle = false;
}

function updateUndoRedoButtons() {
  const undoBtn = document.getElementById("undo-btn");
  const redoBtn = document.getElementById("redo-btn");
  const clearBtn = document.getElementById("clear-btn"); // Get the clear button

  undoBtn.disabled = historyStack.length <= 1;
  redoBtn.disabled = futureStack.length === 0;
  if (clearBtn) {
    // Check if clear button exists
    const editor = document.getElementById("editor");
    clearBtn.disabled = !editor || !editor.value.trim(); // Disable if editor is empty
  }
}

// Load last saved content
window.addEventListener("DOMContentLoaded", () => {
  const localStorageKey = "vibe-text-edits";
  const allEdits = JSON.parse(localStorage.getItem(localStorageKey) || "[]");

  if (allEdits?.length) {
    const editor = document.getElementById("editor");
    if (editor) {
      editor.value = allEdits[allEdits.length - 1].edits; // Load edits from the last object
      // TODO: Optionally, save this loaded state as the initial history state
      // historyStack = [editor.value];
      // updateUndoRedoButtons();
    }
  }
});

/**
 * Initializes the editor functionality
 */
function initEditor() {
  initUndoRedo();

  // Editor setup
  const editor = document.getElementById("editor");
  if (!editor) return; // Exit if editor doesn't exist

  editor.focus();
  editor.addEventListener("input", () => {
    editor.toggleAttribute("data-empty", !editor.value.trim()); // Changed from textContent
    updateUndoRedoButtons(); // Update clear button state on input
  });
  // Initial check for clear button state
  updateUndoRedoButtons();
}

/**
 * Attaches event listeners to DOM elements
 */
function attachEventListeners() {
  const copyButton = document.getElementById("sticky-copy");
  const saveButton = document.getElementById("sticky-save");
  const styleSelector = document.getElementById("style-selector");
  const editor = document.getElementById("editor");
  const clearButton = document.getElementById("clear-btn");
  const orderedListButton = document.getElementById("ordered-list-btn");
  const unorderedListButton = document.getElementById("unordered-list-btn");
  const unicodeQuoteButton = document.getElementById("unicode-quote-btn");

  if (!editor) return; // Exit if editor doesn't exist for event listeners that need it

  if (clearButton) {
    clearButton.addEventListener("click", () => {
      if (editor) {
        editor.value = ""; // Clear the editor content
        saveState(); // This will add the empty state to history and clear redo stack.
        updateUndoRedoButtons(); // Update all button states
        editor.focus();
        showTooltip("clear-btn", "Cleared!");
      }
    });
  }

  if (copyButton) {
    copyButton.addEventListener("click", async () => {
      if (!editor.value.trim()) return; // Changed from innerText

      try {
        await navigator.clipboard.writeText(editor.value); // Changed from innerText
        showTooltip("sticky-copy", "Copied!");
      } catch (err) {
        showTooltip("sticky-copy", "Failed!");
        console.error("Copy failed:", err);
      }
    });
  }

  if (saveButton) {
    saveButton.addEventListener("click", () => {
      const editor = document.getElementById("editor");
      if (!editor) return;

      const contentToSave = editor.value;
      const localStorageKey = "vibe-text-edits";

      try {
        let allEdits = JSON.parse(
          localStorage.getItem(localStorageKey) || "[]"
        );

        const newId =
          allEdits.length > 0 ? Math.max(...allEdits.map((e) => e.id)) + 1 : 1;
        const newEdit = {
          id: newId,
          edits: contentToSave,
          time: new Date().toISOString(),
        };

        allEdits.push(newEdit);
        localStorage.setItem(localStorageKey, JSON.stringify(allEdits));

        if (contentToSave.trim() === "") {
          showTooltip("sticky-save", "Blank Page Saved!");
        } else {
          showTooltip("sticky-save", "Saved!");
        }
      } catch (err) {
        console.error("Save failed:", err);
        showTooltip("sticky-save", "Failed!");
      }
    });
  }

  if (styleSelector) {
    styleSelector.addEventListener("change", (e) => {
      const style = e.target.value;
      if (style) {
        // Visual feedback
        e.target.classList.add("ring-1", "ring-green-400");
        setTimeout(() => {
          e.target.classList.remove("ring-1", "ring-green-400");
          e.target.value = ""; // Reset the select
        }, 200);
        applyStyle(style); // Apply the selected style
      }
    });
  }

  if (orderedListButton) {
    orderedListButton.addEventListener("click", () => {
      applyListFormatting("ordered");
    });
  }

  if (unorderedListButton) {
    unorderedListButton.addEventListener("click", () => {
      applyListFormatting("unordered");
    });
  }

  if (unicodeQuoteButton) {
    unicodeQuoteButton.addEventListener("click", () => {
      applyUnicodeQuote();
    });
  }
}

// New function to apply Unicode quotes
function applyUnicodeQuote() {
  const editor = document.getElementById("editor");
  if (!editor) return;

  const selectionStart = editor.selectionStart;
  const selectionEnd = editor.selectionEnd;
  const currentText = editor.value;

  if (selectionStart === selectionEnd) {
    showTooltip("unicode-quote-btn", "Select text to quote.");
    editor.focus();
    return;
  }

  saveState(); // Save state before applying formatting
  isApplyingStyle = true;

  const selectedText = currentText.substring(selectionStart, selectionEnd);
  const quotedText = `\u275D${selectedText}\u275E`; // Using ❝ and ❞

  editor.value =
    currentText.substring(0, selectionStart) +
    quotedText +
    currentText.substring(selectionEnd);

  // Restore selection to cover the new quoted text
  editor.setSelectionRange(selectionStart, selectionStart + quotedText.length);

  isApplyingStyle = false;
  saveState(); // Save state after applying formatting
  updateUndoRedoButtons();
  showTooltip("unicode-quote-btn", "Quote applied!");
  editor.focus();
}

// =============================================
// Settings Modal Logic
// =============================================
const DISABLED_SITES_STORAGE_KEY = "viblifyDisabledSites";
const SAVED_CONTENT_STORAGE_KEY = "vibe-text-edits"; // Existing key

function initSettingsModal() {
  const settingsModal = document.getElementById("settings-modal");
  const settingsBtn = document.getElementById("settings-btn");
  const closeSettingsModalBtn = document.getElementById("close-settings-modal");

  const disabledSiteInput = document.getElementById("disabled-site-input");
  const addDisabledSiteBtn = document.getElementById("add-disabled-site-btn");
  const disabledSitesListUI = document.getElementById("disabled-sites-list");

  const clearAllSavedContentBtn = document.getElementById(
    "clear-all-saved-content-btn"
  );
  const savedContentListUI = document.getElementById("saved-content-list");
  const editor = document.getElementById("editor");

  if (!settingsModal || !settingsBtn || !closeSettingsModalBtn || !editor) {
    console.warn(
      "Settings modal elements not found. Skipping settings initialization."
    );
    return;
  }

  function toggleSettingsModal(show) {
    if (show) {
      settingsModal.classList.remove("hidden");
      // Tailwind uses 'items-center' and 'justify-center' which imply flex.
      // Re-adding flex explicitly if it was removed by 'hidden' or other means.
      settingsModal.style.display = "flex";
      loadAndRenderDisabledSites();
      loadAndRenderSavedContent();
    } else {
      settingsModal.classList.add("hidden");
      settingsModal.style.display = "none";
    }
  }

  settingsBtn.addEventListener("click", () => toggleSettingsModal(true));
  closeSettingsModalBtn.addEventListener("click", () =>
    toggleSettingsModal(false)
  );
  // Close modal if clicking outside of it
  settingsModal.addEventListener("click", (event) => {
    if (event.target === settingsModal) {
      toggleSettingsModal(false);
    }
  });

  // --- Disabled Websites Logic ---
  async function getDisabledSites() {
    return new Promise((resolve) => {
      chrome.storage.sync.get([DISABLED_SITES_STORAGE_KEY], (result) => {
        resolve(result[DISABLED_SITES_STORAGE_KEY] || []);
      });
    });
  }

  async function saveDisabledSites(sites) {
    return new Promise((resolve) => {
      chrome.storage.sync.set({ [DISABLED_SITES_STORAGE_KEY]: sites }, () => {
        resolve();
        showTooltip(addDisabledSiteBtn.id, "Settings saved!"); // Optional feedback
      });
    });
  }

  function renderDisabledSites(sites) {
    disabledSitesListUI.innerHTML = ""; // Clear existing list
    if (!sites || sites.length === 0) {
      const li = document.createElement("li");
      li.textContent = "No websites disabled yet.";
      li.className = "text-zinc-500 text-sm italic";
      disabledSitesListUI.appendChild(li);
      return;
    }
    sites.forEach((site) => {
      const li = document.createElement("li");
      li.className =
        "flex justify-between items-center bg-zinc-700 px-3 py-2 rounded-md text-sm text-zinc-300";

      const siteText = document.createElement("span");
      siteText.textContent = site;
      li.appendChild(siteText);

      const removeBtn = document.createElement("button");
      removeBtn.className = "text-red-400 hover:text-red-500";
      removeBtn.innerHTML = '<i class="fas fa-trash-alt"></i>';
      removeBtn.title = `Remove ${site}`;
      removeBtn.addEventListener("click", async () => {
        const currentSites = await getDisabledSites();
        const updatedSites = currentSites.filter((s) => s !== site);
        await saveDisabledSites(updatedSites);
        renderDisabledSites(updatedSites);
        showTooltip(
          removeBtn.parentElement.parentElement.id,
          `Removed ${site}!`
        ); // Tooltip on the list
      });
      li.appendChild(removeBtn);
      disabledSitesListUI.appendChild(li);
    });
  }

  async function loadAndRenderDisabledSites() {
    const sites = await getDisabledSites();
    renderDisabledSites(sites);
  }

  addDisabledSiteBtn.addEventListener("click", async () => {
    const siteToAdd = disabledSiteInput.value.trim().toLowerCase();
    if (siteToAdd) {
      // Basic validation: remove http(s):// and www.
      let domain = siteToAdd;
      try {
        const url = new URL(
          siteToAdd.startsWith("http") ? siteToAdd : `http://${siteToAdd}`
        );
        domain = url.hostname.replace(/^www\./, "");
      } catch (e) {
        // If not a valid URL, use as is, but good to clean common prefixes
        domain = siteToAdd.replace(/^(https?:\/\/)?(www\.)?/, "").split("/")[0];
      }

      if (!domain) {
        showTooltip(addDisabledSiteBtn.id, "Invalid domain!");
        return;
      }

      const currentSites = await getDisabledSites();
      if (!currentSites.includes(domain)) {
        const updatedSites = [...currentSites, domain];
        await saveDisabledSites(updatedSites);
        renderDisabledSites(updatedSites);
        disabledSiteInput.value = "";
        showTooltip(addDisabledSiteBtn.id, `${domain} added!`);
      } else {
        showTooltip(addDisabledSiteBtn.id, `${domain} already listed!`);
      }
      disabledSiteInput.value = ""; // Clear input regardless
    } else {
      showTooltip(addDisabledSiteBtn.id, "Enter a domain!");
    }
  });

  // --- Saved Content Logic ---
  function getSavedContent() {
    return JSON.parse(localStorage.getItem(SAVED_CONTENT_STORAGE_KEY) || "[]");
  }

  function renderSavedContent(items) {
    savedContentListUI.innerHTML = ""; // Clear existing list
    if (!items || items.length === 0) {
      const li = document.createElement("li");
      li.textContent = "No content saved yet.";
      li.className = "text-zinc-500 text-sm italic";
      savedContentListUI.appendChild(li);
      return;
    }
    // Show newest first
    items
      .slice()
      .reverse()
      .forEach((item) => {
        const li = document.createElement("li");
        li.className =
          "flex justify-between items-center bg-zinc-700 px-3 py-2 rounded-md text-sm text-zinc-300";

        const itemInfo = document.createElement("span");
        const date = new Date(item.time).toLocaleDateString("en-US", {
          year: "2-digit",
          month: "short",
          day: "numeric",
        });
        const preview =
          item.edits.substring(0, 20) + (item.edits.length > 20 ? "..." : "");
        itemInfo.textContent = `"${preview}" (${date})`;
        itemInfo.title = item.edits; // Full text on hover
        li.appendChild(itemInfo);

        const loadBtn = document.createElement("button");
        loadBtn.className =
          "text-green-400 hover:text-green-500 ml-2 px-2 py-1 rounded hover:bg-zinc-600 transition-colors";
        loadBtn.innerHTML = '<i class="fas fa-upload mr-1"></i> Load';
        loadBtn.addEventListener("click", () => {
          editor.value = item.edits;
          saveState(); // Update history for undo/redo
          updateUndoRedoButtons();
          showTooltip(
            loadBtn.parentElement.parentElement.id,
            "Content loaded to editor!"
          ); // Tooltip on the list
          toggleSettingsModal(false); // Close modal after loading
        });
        li.appendChild(loadBtn);
        savedContentListUI.appendChild(li);
      });
  }

  async function loadAndRenderSavedContent() {
    const items = getSavedContent();
    renderSavedContent(items);
  }

  clearAllSavedContentBtn.addEventListener("click", () => {
    if (
      confirm(
        "Are you sure you want to delete ALL saved snippets? This cannot be undone."
      )
    ) {
      localStorage.removeItem(SAVED_CONTENT_STORAGE_KEY);
      loadAndRenderSavedContent(); // Re-render (will show empty)
      showTooltip(clearAllSavedContentBtn.id, "All saved snippets cleared!");
    }
  });

  // Initial population when modal is first configured (though it reloads on open)
  // loadAndRenderDisabledSites();
  // loadAndRenderSavedContent();
}

// Initialize the main application
// ...
window.addEventListener("DOMContentLoaded", () => {
  const localStorageKey = "vibe-text-edits"; // This is SAVED_CONTENT_STORAGE_KEY
  const allEdits = JSON.parse(localStorage.getItem(localStorageKey) || "[]");

  if (allEdits?.length) {
    const editor = document.getElementById("editor");
    if (editor) {
      editor.value = allEdits[allEdits.length - 1].edits; // Load edits from the last object
      // TODO: Optionally, save this loaded state as the initial history state
      // historyStack = [editor.value];
      // updateUndoRedoButtons();
    }
  }
  initEditor();
  attachEventListeners();
  initSettingsModal(); // Initialize settings modal functionality
});

function copyToClipboard(text, buttonId, message) {
  navigator.clipboard
    .writeText(text)
    .then(() => {
      showTooltip(buttonId, message);
    })
    .catch((err) => {
      console.error("Failed to copy:", err);
      showTooltip(buttonId, "Copy failed!");
    });
}
