function initUnicodeSuggestions() {
  const MIN_INPUT_HEIGHT = 20;
  const MIN_INPUT_WIDTH = 100;
  const IDLE_TIMEOUT = 25000;

  // Helper function to detect Mac OS
  const isMac = () => navigator.platform.toUpperCase().includes("MAC");

  // Debounce function for performance
  function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  }

  // Text sample for previews
  let previewText = "Abc";

  // Active style layers
  let activeLayers = [];

  // Site-specific theme detection
  const SITE_THEMES = [
    {
      name: "twitter",
      patterns: ["twitter.com", "x.com"],
      primaryColor: "#1da1f2",
    },
    {
      name: "facebook",
      patterns: ["facebook.com", "fb.com"],
      primaryColor: "#4267B2",
    },
    {
      name: "linkedin",
      patterns: ["linkedin.com"],
      primaryColor: "#0077b5",
    },
    {
      name: "instagram",
      patterns: ["instagram.com"],
      primaryColor: "#e1306c",
    },
    {
      name: "reddit",
      patterns: ["reddit.com"],
      primaryColor: "#ff5600",
    },
    {
      name: "github",
      patterns: ["github.com"],
      primaryColor: "#24292e",
    },
  ];

  // Currency categories
  const CURRENCY_CATEGORIES = [
    {
      name: "Popular",
      symbols: [
        { id: "usd", name: "US Dollar", symbol: "$" },
        { id: "eur", name: "Euro", symbol: "€" },
        { id: "gbp", name: "British Pound", symbol: "£" },
        { id: "jpy", name: "Japanese Yen", symbol: "¥" },
        { id: "inr", name: "Indian Rupee", symbol: "₹" },
        { id: "btc", name: "Bitcoin", symbol: "₿" },
        { id: "rub", name: "Russian Ruble", symbol: "₽" },
        { id: "krw", name: "Korean Won", symbol: "₩" },
      ],
    },
    {
      name: "Other",
      symbols: [
        { id: "baht", name: "Thai Baht", symbol: "฿" },
        { id: "ngn", name: "Nigerian Naira", symbol: "₦" },
        { id: "peso", name: "Philippine Peso", symbol: "₱" },
        { id: "brazil", name: "Brazilian Real", symbol: "R$" },
        { id: "lira", name: "Turkish Lira", symbol: "₺" },
        { id: "cent", name: "Cent", symbol: "¢" },
        { id: "genericCurrency", name: "Currency", symbol: "¤" },
      ],
    },
  ];

  // Special characters categories
  const SPECIAL_CHAR_CATEGORIES = [
    {
      name: "Yoruba",
      chars: [
        { char: "ẹ", name: "e with dot below" },
        { char: "Ẹ", name: "E with dot below" },
        { char: "ọ", name: "o with dot below" },
        { char: "Ọ", name: "O with dot below" },
        { char: "ṣ", name: "s with dot below" },
        { char: "Ṣ", name: "S with dot below" },
        { char: "ẹ́", name: "e with dot below and acute" },
        { char: "Ẹ́", name: "E with dot below and acute" },
        { char: "ọ́", name: "o with dot below and acute" },
        { char: "Ọ́", name: "O with dot below and acute" },
        { char: "ṣ́", name: "s with dot below and acute" },
        { char: "Ṣ́", name: "S with dot below and acute" },
      ],
    },
    {
      name: "Eastern European",
      chars: [
        { char: "č", name: "c with caron" },
        { char: "Č", name: "C with caron" },
        { char: "š", name: "s with caron" },
        { char: "Š", name: "S with caron" },
        { char: "ž", name: "z with caron" },
        { char: "Ž", name: "Z with caron" },
        { char: "đ", name: "d with stroke" },
        { char: "Đ", name: "D with stroke" },
        { char: "ć", name: "c with acute" },
        { char: "Ć", name: "C with acute" },
        { char: "ň", name: "n with caron" },
        { char: "Ň", name: "N with caron" },
        { char: "ř", name: "r with caron" },
        { char: "Ř", name: "R with caron" },
        { char: "ľ", name: "l with caron" },
        { char: "Ľ", name: "L with caron" },
        { char: "ť", name: "t with caron" },
        { char: "Ť", name: "T with caron" },
      ],
    },
    {
      name: "Latin Extended",
      chars: [
        { char: "ñ", name: "n with tilde" },
        { char: "Ñ", name: "N with tilde" },
        { char: "á", name: "a with acute" },
        { char: "Á", name: "A with acute" },
        { char: "é", name: "e with acute" },
        { char: "É", name: "E with acute" },
        { char: "í", name: "i with acute" },
        { char: "Í", name: "I with acute" },
        { char: "ó", name: "o with acute" },
        { char: "Ó", name: "O with acute" },
        { char: "ú", name: "u with acute" },
        { char: "Ú", name: "U with acute" },
        { char: "à", name: "a with grave" },
        { char: "À", name: "A with grave" },
        { char: "è", name: "e with grave" },
        { char: "È", name: "E with grave" },
        { char: "ì", name: "i with grave" },
        { char: "Ì", name: "I with grave" },
        { char: "ò", name: "o with grave" },
        { char: "Ò", name: "O with grave" },
        { char: "ù", name: "u with grave" },
        { char: "Ù", name: "U with grave" },
        { char: "ä", name: "a with diaeresis" },
        { char: "Ä", name: "A with diaeresis" },
        { char: "ë", name: "e with diaeresis" },
        { char: "Ë", name: "E with diaeresis" },
        { char: "ï", name: "i with diaeresis" },
        { char: "Ï", name: "I with diaeresis" },
        { char: "ö", name: "o with diaeresis" },
        { char: "Ö", name: "O with diaeresis" },
        { char: "ü", name: "u with diaeresis" },
        { char: "Ü", name: "U with diaeresis" },
        { char: "â", name: "a with circumflex" },
        { char: "Â", name: "A with circumflex" },
        { char: "ê", name: "e with circumflex" },
        { char: "Ê", name: "E with circumflex" },
        { char: "î", name: "i with circumflex" },
        { char: "Î", name: "I with circumflex" },
        { char: "ô", name: "o with circumflex" },
        { char: "Ô", name: "O with circumflex" },
        { char: "û", name: "u with circumflex" },
        { char: "Û", name: "U with circumflex" },
        { char: "ã", name: "a with tilde" },
        { char: "Ã", name: "A with tilde" },
        { char: "õ", name: "o with tilde" },
        { char: "Õ", name: "O with tilde" },
        { char: "ç", name: "c with cedilla" },
        { char: "Ç", name: "C with cedilla" },
      ],
    },
    {
      name: "Mathematical & Symbols",
      chars: [
        { char: "±", name: "plus-minus" },
        { char: "×", name: "multiplication" },
        { char: "÷", name: "division" },
        { char: "≠", name: "not equal" },
        { char: "≤", name: "less than or equal" },
        { char: "≥", name: "greater than or equal" },
        { char: "≈", name: "approximately equal" },
        { char: "∞", name: "infinity" },
        { char: "π", name: "pi" },
        { char: "α", name: "alpha" },
        { char: "β", name: "beta" },
        { char: "γ", name: "gamma" },
        { char: "θ", name: "theta" },
        { char: "λ", name: "lambda" },
        { char: "µ", name: "mu" },
        { char: "σ", name: "sigma" },
        { char: "φ", name: "phi" },
        { char: "ω", name: "omega" },
        { char: "•", name: "bullet" },
        { char: "‰", name: "per mille" },
        { char: "§", name: "section" },
        { char: "†", name: "dagger" },
        { char: "«", name: "left double angle quotation" },
        { char: "»", name: "right double angle quotation" },
        { char: "—", name: "em dash" },
        { char: "–", name: "en dash" },
        { char: "…", name: "ellipsis" },
        { char: "™", name: "trademark" },
        { char: "®", name: "registered" },
        { char: "©", name: "copyright" },
        { char: "°", name: "degree" },
        { char: "¼", name: "one quarter" },
        { char: "½", name: "one half" },
        { char: "¾", name: "three quarters" },
      ],
    },
  ];

  // Define custom handler functions before STYLES array
  function applyListFormatting(inputElement, listType) {
    const isTextareaOrInput =
      inputElement.tagName === "TEXTAREA" || inputElement.tagName === "INPUT";
    let originalText, selectionStart, selectionEnd;

    if (isTextareaOrInput) {
      originalText = inputElement.value;
      selectionStart = inputElement.selectionStart;
      selectionEnd = inputElement.selectionEnd;
    } else if (inputElement.isContentEditable) {
      const selection = window.getSelection();
      if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        originalText = range.toString();
        // For contentEditable, complex range manipulation is needed to replace selected lines.
        // This simplified version will format the selected block or the whole content if no selection.
        if (range.collapsed) originalText = inputElement.textContent;
      } else {
        originalText = inputElement.textContent; // Whole content
      }
    } else {
      console.warn("applyListFormatting: Unsupported input element type");
      return;
    }

    const textToFormat =
      isTextareaOrInput && selectionStart !== selectionEnd
        ? originalText.substring(selectionStart, selectionEnd)
        : originalText;
    const lines = textToFormat.split("\n");
    let itemCounter = 1;
    const newListLines = lines.map((line) => {
      if (line.trim() === "") return line;
      if (listType === "orderedList") {
        return `${itemCounter++}. ${line}`;
      } else {
        // unorderedList
        return `• ${line}`;
      }
    });
    const formattedText = newListLines.join("\n");

    if (isTextareaOrInput) {
      if (selectionStart !== selectionEnd) {
        inputElement.value =
          originalText.substring(0, selectionStart) +
          formattedText +
          originalText.substring(selectionEnd);
        inputElement.selectionStart = selectionStart;
        inputElement.selectionEnd = selectionStart + formattedText.length;
      } else {
        inputElement.value = formattedText; // Or apply to relevant lines around cursor
      }
    } else if (inputElement.isContentEditable) {
      const selection = window.getSelection();
      if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        if (!range.collapsed) {
          range.deleteContents();
          range.insertNode(document.createTextNode(formattedText));
        } else {
          inputElement.textContent = formattedText; // Replace whole content
        }
      } else {
        inputElement.textContent = formattedText;
      }
    }
    inputElement.dispatchEvent(new Event("input", { bubbles: true }));
  }

  function applyUnicodeQuote(inputElement) {
    const isTextareaOrInput =
      inputElement.tagName === "TEXTAREA" || inputElement.tagName === "INPUT";
    let newTextValue;

    if (isTextareaOrInput) {
      const selectionStart = inputElement.selectionStart;
      const selectionEnd = inputElement.selectionEnd;
      const selectedText = inputElement.value.substring(
        selectionStart,
        selectionEnd
      );

      if (selectedText) {
        const before = inputElement.value.substring(0, selectionStart);
        const after = inputElement.value.substring(selectionEnd);
        newTextValue = `${before}\u275D${selectedText}\u275E${after}`;
        inputElement.value = newTextValue;
        inputElement.selectionStart = selectionStart + 1;
        inputElement.selectionEnd = selectionEnd + 1;
      } else {
        // Wrap whole content if no selection
        newTextValue = `\u275D${inputElement.value}\u275E`;
        inputElement.value = newTextValue;
      }
    } else if (inputElement.isContentEditable) {
      const selection = window.getSelection();
      if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        if (!range.collapsed) {
          const selectedText = range.toString();
          const quotedTextNode = document.createTextNode(
            `\u275D${selectedText}\u275E`
          );
          range.deleteContents();
          range.insertNode(quotedTextNode);
          range.selectNodeContents(quotedTextNode); // Select the new text
          selection.removeAllRanges();
          selection.addRange(range);
        } else {
          // Wrap whole content if no selection (simplified)
          inputElement.textContent = `\u275D${inputElement.textContent}\u275E`;
        }
      } else {
        inputElement.textContent = `\u275D${inputElement.textContent}\u275E`;
      }
    } else {
      console.warn("applyUnicodeQuote: Unsupported input element type");
      return;
    }
    inputElement.dispatchEvent(new Event("input", { bubbles: true }));
  }

  const STYLES = [
    {
      id: "bold",
      name: "Bold",
      char: "𝗕",
      cmd: "bold",
      shortcut: ["Ctrl+B", "Cmd+B"],
    },
    {
      id: "italic",
      name: "Italic",
      char: "𝘪",
      cmd: "italic",
      shortcut: ["Ctrl+I", "Cmd+I"],
    },
    {
      id: "script",
      name: "Script",
      char: "𝓢",
      cmd: "script",
      shortcut: ["Ctrl+S", "Cmd+S"],
    },
    {
      id: "monospace",
      name: "Monospace",
      char: "𝙼",
      cmd: "monospace",
      shortcut: ["Ctrl+M", "Cmd+M"],
    },
    {
      id: "fraktur",
      name: "Fraktur",
      char: "𝔉",
      cmd: "fraktur",
      shortcut: ["Ctrl+F", "Cmd+F"],
    },
    {
      id: "doubleStruck",
      name: "Double Struck",
      char: "𝕊",
      cmd: "doubleStruck",
      shortcut: ["Ctrl+D", "Cmd+D"],
    },
    {
      id: "strikethrough",
      name: "Strikethrough",
      char: "S̶", // Using S with combining long stroke overlay as example
      cmd: "strikethrough",
      shortcut: ["Ctrl+T", "Cmd+T"], // Assigning a new shortcut, ensure it's unique
    },
    {
      id: "unorderedList",
      name: "Unordered List",
      char: "•",
      cmd: "unorderedList",
      shortcut: ["", ""],
      applyFunction: applyListFormatting, // Add reference to a function
    },
    {
      id: "orderedList",
      name: "Ordered List",
      char: "1.",
      cmd: "orderedList",
      shortcut: ["", ""],
      applyFunction: applyListFormatting, // Add reference to a function
    },
    {
      id: "unicodeQuote",
      name: "Quote",
      char: "❝❞",
      cmd: "unicodeQuote",
      shortcut: ["", ""],
      applyFunction: applyUnicodeQuote, // Add reference to a function
    },
  ];

  // Track active popup
  let activePopup = null;
  let activeInput = null; // Keep track of the currently focused input
  let inputCounter = 0; // Counter for unique IDs

  // Track active tab
  let activeTab = "styles"; // 'styles', 'currency', or 'special'

  // Detect site and apply appropriate theme
  function detectSiteTheme() {
    const currentUrl = window.location.href;
    let detectedSite = null;

    for (const site of SITE_THEMES) {
      for (const pattern of site.patterns) {
        if (currentUrl.includes(pattern)) {
          detectedSite = site.name;
          break;
        }
      }
      if (detectedSite) break;
    }

    return detectedSite;
  }

  function applyTheme(theme) {
    document.body.classList.toggle("unicode-dark", theme === "dark");
    document.body.classList.toggle("unicode-light", theme === "light");

    // Remove any existing site theme classes
    SITE_THEMES.forEach((site) => {
      document.body.classList.remove(`site-${site.name}`);
    });

    // Apply site-specific theme if detected
    const siteTheme = detectSiteTheme();
    if (siteTheme) {
      document.body.classList.add(`site-${siteTheme}`);
    }

    const popups = document.querySelectorAll(".unicode-suggestion-popup");
    popups.forEach((popup) => {
      popup.classList.toggle("dark", theme === "dark");
      popup.classList.toggle("light", theme === "light");
    });
  }

  function loadTheme(callback) {
    chrome.storage.sync.get("theme", (data) => {
      const theme = data.theme || "dark"; // Default to dark to match your original CSS
      applyTheme(theme);
      callback(theme);
    });
  }

  function toggleTheme(currentTheme) {
    const newTheme = currentTheme === "dark" ? "light" : "dark";
    chrome.storage.sync.set({ theme: newTheme }, () => applyTheme(newTheme));
    return newTheme;
  }

  // Apply multiple style transformations for style layers
  function applyLayeredStyles(text, stylesList) {
    if (!stylesList || !stylesList.length) return text;

    let result = text;
    for (const cmd of stylesList) {
      // cmd is the command string, e.g., "bold" or "unorderedList"
      const styleObject = STYLES.find((s) => s.cmd === cmd);

      // If the style has a dedicated applyFunction, it's an action, not a simple text transformation
      // that uses styleMaps. These should be handled by their applyFunction directly when invoked,
      // not as part of a layered text transformation sequence.
      // If such a command reaches here via activeLayers, we skip it to prevent errors.
      if (styleObject && typeof styleObject.applyFunction === "function") {
        continue; // Skip this command in the layering chain
      }
      result = convertText(result, cmd); // Only call convertText for styles that use styleMaps
    }
    return result;
  }

  function convertText(text, style) {
    const styleMap = window.UnicodeStyler?.styleMaps[style];
    if (!styleMap) {
      return text;
    }

    return text
      .split("")
      .map((char) => {
        if (styleMap[char]) return styleMap[char];
        const oppositeCase =
          char === char.toLowerCase() ? char.toUpperCase() : char.toLowerCase();
        return styleMap[oppositeCase] || char;
      })
      .join("");
  }

  function applyStyle(input, styleOrCmd, isLayer = false) {
    if (!input || !(input instanceof HTMLElement)) {
      console.error("Invalid input element");
      return;
    }

    // Find the style object from STYLES array if a command string is passed
    const styleObject =
      typeof styleOrCmd === "string"
        ? STYLES.find((s) => s.cmd === styleOrCmd)
        : null;

    // If a specific applyFunction exists for the style, use it
    if (styleObject && typeof styleObject.applyFunction === "function") {
      styleObject.applyFunction(input, styleObject.cmd); // Pass input and command
      return;
    }

    // Fallback to existing style application logic for text transformations
    let styleToApply = styleOrCmd;
    if (isLayer) {
      const layerIndex = activeLayers.indexOf(styleToApply);
      if (layerIndex >= 0) {
        // Remove this layer
        activeLayers.splice(layerIndex, 1);
      } else {
        // Add this layer
        activeLayers.push(styleToApply);
      }
      // Use all active layers to transform text
      styleToApply = activeLayers.length > 0 ? activeLayers : null;
    }

    const twitterEditor = input.closest('[data-testid="tweetTextarea_0"]');
    if (twitterEditor) {
      applyTwitterStyle(twitterEditor, styleToApply, isLayer);
      return;
    }

    if (input.tagName === "INPUT" || input.tagName === "TEXTAREA") {
      applyStandardInputStyle(input, styleToApply, isLayer);
      return;
    }

    if (input.isContentEditable) {
      applyContentEditableStyle(input, styleToApply, isLayer);
    }
  }

  function applyTwitterStyle(editor, style, isLayer = false) {
    const selection = window.getSelection();
    if (!selection.rangeCount) return;

    const range = selection.getRangeAt(0);
    const contentDiv = editor.querySelector('[data-contents="true"]');

    if (!contentDiv) {
      console.error("Could not find Twitter content div");
      return;
    }

    try {
      // Get current cursor position to restore later
      const selectionState = saveTwitterSelection(contentDiv);

      if (!range.collapsed) {
        const selectedText = range.toString();
        if (selectedText) {
          let styledText;
          if (isLayer && Array.isArray(style)) {
            styledText = applyLayeredStyles(selectedText, style);
          } else if (!isLayer) {
            styledText = convertText(selectedText, style);
          } else {
            return; // No styles to apply
          }

          range.deleteContents();
          range.insertNode(document.createTextNode(styledText));
          triggerTwitterEvents(editor);

          // Restore selection
          restoreTwitterSelection(contentDiv, selectionState);
          return;
        }
      }

      const textNodes = getTextNodes(contentDiv);
      textNodes.forEach((node) => {
        if (isLayer && Array.isArray(style)) {
          node.nodeValue = applyLayeredStyles(node.nodeValue, style);
        } else if (!isLayer) {
          node.nodeValue = convertText(node.nodeValue, style);
        }
      });

      triggerTwitterEvents(editor);

      // Restore selection after applying style
      restoreTwitterSelection(contentDiv, selectionState);

      // Ensure editor maintains focus
      setTimeout(() => {
        editor.focus();
      }, 10);
    } catch (error) {
      console.error("Error applying Twitter style:", error);
    }
  }

  // Helper functions for Twitter selection preservation
  function saveTwitterSelection(containerEl) {
    const selection = window.getSelection();
    if (selection.rangeCount > 0) {
      const range = selection.getRangeAt(0);
      const preSelectionRange = range.cloneRange();
      preSelectionRange.selectNodeContents(containerEl);
      preSelectionRange.setEnd(range.startContainer, range.startOffset);
      const start = preSelectionRange.toString().length;

      return {
        start: start,
        end: start + range.toString().length,
      };
    }
    return null;
  }

  function restoreTwitterSelection(containerEl, savedSel) {
    if (!savedSel) return;

    const selection = window.getSelection();
    const range = document.createRange();
    range.setStart(containerEl, 0);
    range.collapse(true);

    const nodeStack = [containerEl];
    let node,
      foundStart = false,
      stop = false;
    let charIndex = 0;

    while (!stop && (node = nodeStack.pop())) {
      if (node.nodeType === Node.TEXT_NODE) {
        const nextCharIndex = charIndex + node.length;
        if (
          !foundStart &&
          savedSel.start >= charIndex &&
          savedSel.start <= nextCharIndex
        ) {
          range.setStart(node, savedSel.start - charIndex);
          foundStart = true;
        }
        if (
          foundStart &&
          savedSel.end >= charIndex &&
          savedSel.end <= nextCharIndex
        ) {
          range.setEnd(node, savedSel.end - charIndex);
          stop = true;
        }
        charIndex = nextCharIndex;
      } else {
        let i = node.childNodes.length;
        while (i--) {
          nodeStack.push(node.childNodes[i]);
        }
      }
    }

    selection.removeAllRanges();
    selection.addRange(range);
  }

  function getTextNodes(element) {
    const textNodes = [];
    const walker = document.createTreeWalker(
      element,
      NodeFilter.SHOW_TEXT,
      null,
      false
    );

    let node;
    while ((node = walker.nextNode())) {
      textNodes.push(node);
    }

    return textNodes;
  }

  function triggerTwitterEvents(editor) {
    const inputEvent = new InputEvent("input", {
      bubbles: true,
      cancelable: true,
    });
    editor.dispatchEvent(inputEvent);

    const changeEvent = new Event("change", {
      bubbles: true,
      cancelable: true,
    });
    editor.dispatchEvent(changeEvent);

    // Don't call blur() as it causes the focus issue
    // Just ensure focus is maintained with setTimeout
    setTimeout(() => {
      editor.focus();
    }, 10);
  }

  function applyStandardInputStyle(input, style, isLayer = false) {
    const start = input.selectionStart;
    const end = input.selectionEnd;
    const text = input.value;

    if (start === end) {
      // Transform whole text
      if (isLayer && Array.isArray(style)) {
        input.value = applyLayeredStyles(text, style);
      } else if (!isLayer) {
        input.value = convertText(text, style);
      }
    } else {
      // Transform selected text
      const selectedText = text.substring(start, end);
      let styledText;

      if (isLayer && Array.isArray(style)) {
        styledText = applyLayeredStyles(selectedText, style);
      } else if (!isLayer) {
        styledText = convertText(selectedText, style);
      } else {
        return; // No styles to apply
      }

      input.value = text.substring(0, start) + styledText + text.substring(end);
    }

    input.dispatchEvent(new Event("input", { bubbles: true }));
  }

  function applyContentEditableStyle(element, style, isLayer = false) {
    const selection = window.getSelection();
    if (!selection.rangeCount) return;

    const range = selection.getRangeAt(0);

    if (!range.collapsed) {
      const selectedText = range.toString();
      let styledText;

      if (isLayer && Array.isArray(style)) {
        styledText = applyLayeredStyles(selectedText, style);
      } else if (!isLayer) {
        styledText = convertText(selectedText, style);
      } else {
        return; // No styles to apply
      }

      range.deleteContents();
      range.insertNode(document.createTextNode(styledText));
    } else {
      if (isLayer && Array.isArray(style)) {
        element.textContent = applyLayeredStyles(element.textContent, style);
      } else if (!isLayer) {
        element.textContent = convertText(element.textContent, style);
      }
    }

    element.dispatchEvent(new Event("input", { bubbles: true }));
  }
  function createSuggestionButton(targetInput) {
    const btn = document.createElement("button");
    btn.className = "unicode-suggestion-btn hidden";

    // Determine button size based on input dimensions
    const rect = targetInput.getBoundingClientRect();
    const isSmallInput = rect.height < 40 || rect.width < 150;

    if (isSmallInput) {
      // Create compact dot version
      btn.classList.add("compact");
      btn.innerHTML = `<div class="viblify-dot"></div>`;
    } else {
      // Create standard version with icon
      btn.innerHTML = `
        <svg class="w-3 h-3" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21a4 4 0 01-4-4V5a2 2 0 012-2h4a2 2 0 012 2v12a4 4 0 01-4 4zm0 0h12a2 2 0 002-2v-4a2 2 0 00-2-2h-2.343M11 7.343l1.657-1.657a2 2 0 012.828 0l2.829 2.829a2 2 0 010 2.828l-8.486 8.485M7 17h.01"/>
        </svg>`;
    }

    // Apply current theme to button
    const currentTheme = document.body.classList.contains("unicode-light") ? "light" : "dark";
    btn.classList.add(`unicode-${currentTheme}`);

    // Add site-specific theme if applicable
    const siteTheme = detectSiteTheme();
    if (siteTheme) {
      btn.classList.add(`site-${siteTheme}`);
    }

    btn.title = "Viblify style";
    btn.setAttribute("aria-label", "Open Unicode text styling options");
    return btn;
  }

  function createSuggestionPopup() {
    const popup = document.createElement("div");
    popup.className = "unicode-suggestion-popup hidden";
    popup.innerHTML = `
      <div class="header flex justify-between items-center p-2 bg-gray-800 dark:bg-gray-900 light:bg-gray-200 rounded-t-md">
        <div class="flex items-center gap-2 text-sm font-medium">
          <span class="text-sm ">Suggestion Styles <br/> <span class="text-neutral-600" style="font-size:10px">(${
            isMac() ? "Cmd" : "Ctrl"
          }+[Key])</span></span>
        </div>
        <div class="flex items-center">
          <button id="theme-toggle" class="cursor-pointer text-xs p-1.5 rounded hover:bg-gray-600 dark:hover:bg-gray-700 light:hover:bg-gray-200">
            <svg class="w-4 h-4 theme-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z"/>
            </svg>
          </button>
        </div>
      </div>
        <div class="tab-navigation">
        <button class="tab-button active" data-tab="styles">Styles</button>
        <button class="tab-button" data-tab="currency">Currency</button>
        <button class="tab-button" data-tab="special" >
          <span style="font-size: 14px;" class="text-sm font-medium">Special</span>
        </button>
      </div>
      <div class="tab-content">
        <div class="tab-pane active" data-tab-content="styles">
          <div class="suggestion-options">
            ${STYLES.map(
              (style) => `
                <button data-cmd="${style.cmd}">
                  <span class="w-6">${style.char}</span> ${
                style.name
              } <span class="ml-2 opacity-70 text-neutral-600 text-[11px]">${
                isMac() ? style.shortcut[1] : style.shortcut[0]
              }</span>
                  <span class="unicode-preview" data-preview-style="${
                    style.cmd
                  }"></span>
                </button>
              `
            ).join("")}
          </div>
        </div>
        <div class="tab-pane" data-tab-content="currency">
          <div class="currency-options">
            ${CURRENCY_CATEGORIES.map(
              (category) => `
              <div class="currency-category">
                <h3 class="currency-category-title">${category.name}</h3>
                ${category.symbols
                  .map(
                    (currency) => `
                  <button data-currency="${currency.symbol}">
                    <span class="w-6 currency-symbol">${currency.symbol}</span> ${currency.name}
                  </button>
                `
                  )
                  .join("")}
              </div>
            `
            ).join("")}
          </div>        </div>
        <div class="tab-pane" data-tab-content="special">
          <div class="special-chars-options">
            ${SPECIAL_CHAR_CATEGORIES.map(
              (category) => `
              <div class="special-chars-category">
                <h3 class="special-chars-category-title">${category.name}</h3>
                <div class="special-chars-grid">
                  ${category.chars
                    .map(
                      (char) => `
                    <button data-special-char="${char.char}" title="${char.name}" class="special-char-btn">
                      ${char.char}
                    </button>
                  `
                    )
                    .join("")}
                </div>
              </div>
            `
            ).join("")}
          </div>
        </div>
    `;

    // Remove layout tab logic (no layout tab)
    // Set up tab switching
    setTimeout(() => {
      const tabButtons = popup.querySelectorAll(".tab-button");
      tabButtons.forEach((button) => {
        button.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();
          const tabName = button.getAttribute("data-tab");
          switchTab(popup, tabName);
        });

        // Prevent tab buttons from taking focus to avoid blur on input
        button.addEventListener("mousedown", (e) => {
          e.preventDefault();
        });
      });
      // Setup layer application buttons
      const applyLayersBtn = popup.querySelector(".layer-apply-btn");
      const clearLayersBtn = popup.querySelector(".layer-clear-btn");
      if (applyLayersBtn) {
        applyLayersBtn.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();
          const targetInput = popup.targetInput;
          if (!targetInput) return;

          const isTwitterEditor =
            targetInput.closest('[data-testid="tweetTextarea_0"]') !== null;

          // Apply all selected layers
          applyStyle(targetInput, activeLayers, true);

          // Update layer previews
          updateLayerPreviews(popup);

          // Focus the editor after applying
          if (isTwitterEditor) {
            setTimeout(() => targetInput.focus(), 20);
          }
        });
      }
      if (clearLayersBtn) {
        clearLayersBtn.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();
          // Clear all checkboxes
          popup
            .querySelectorAll(".style-layer-checkbox")
            .forEach((checkbox) => {
              checkbox.checked = false;
            });

          // Clear active layers
          activeLayers = [];

          // Update previews
          updateLayerPreviews(popup);
        });
      }

      // Setup layer checkbox handling
      popup.querySelectorAll(".style-layer-checkbox").forEach((checkbox) => {
        checkbox.addEventListener("change", () => {
          const layerStyle = checkbox.getAttribute("data-layer");
          const layerIndex = activeLayers.indexOf(layerStyle);

          if (checkbox.checked && layerIndex === -1) {
            activeLayers.push(layerStyle);
          } else if (!checkbox.checked && layerIndex !== -1) {
            activeLayers.splice(layerIndex, 1);
          }

          // Update layer previews
          updateLayerPreviews(popup);
        });
      });
    }, 0);

    // Keyboard navigation for suggestion popup
    popup.addEventListener("keydown", (e) => {
      const visibleTab = popup.querySelector(".tab-pane.active");
      const options = Array.from(visibleTab.querySelectorAll("button"));
      if (!options.length) return;
      let idx = options.findIndex((btn) =>
        btn.classList.contains("highlighted")
      );
      if (e.key === "ArrowDown") {
        e.preventDefault();
        if (idx >= 0) options[idx].classList.remove("highlighted");
        idx = (idx + 1) % options.length;
        options[idx].classList.add("highlighted");
        options[idx].focus();
      } else if (e.key === "ArrowUp") {
        e.preventDefault();
        if (idx >= 0) options[idx].classList.remove("highlighted");
        idx = (idx - 1 + options.length) % options.length;
        options[idx].classList.add("highlighted");
        options[idx].focus();
      } else if (e.key === "Enter") {
        e.preventDefault();
        if (idx >= 0) options[idx].click();
      }
    });

    return popup;
  }

  // Function to update layer previews
  function updateLayerPreviews(popup) {
    const sampleText = previewText || "Abc";

    // Update individual layer previews
    popup.querySelectorAll("[data-layer-preview]").forEach((preview) => {
      const layerStyle = preview.getAttribute("data-layer-preview");
      preview.textContent = convertText(sampleText, layerStyle);
    });

    // If we have active layers, show a combined preview somewhere
    if (activeLayers.length > 0) {
      const combinedPreview = applyLayeredStyles(sampleText, activeLayers);
      // You can add a specific combined preview element if desired
      // Or just update something like a status text
    }
  }

  // Function to switch between tabs
  function switchTab(popup, tabName) {
    activeTab = tabName;

    // Update tab buttons
    const tabButtons = popup.querySelectorAll(".tab-button");
    tabButtons.forEach((button) => {
      if (button.getAttribute("data-tab") === tabName) {
        button.classList.add("active");
      } else {
        button.classList.remove("active");
      }
    });

    // Update tab content
    const tabPanes = popup.querySelectorAll(".tab-pane");
    tabPanes.forEach((pane) => {
      if (pane.getAttribute("data-tab-content") === tabName) {
        pane.classList.add("active");
      } else {
        pane.classList.remove("active");
      }
    });

    // Update previews for specific tabs
    if (tabName === "layers") {
      updateLayerPreviews(popup);
    }
  }

  // Enhanced smart positioning that avoids UI elements
  function smartPositionPopup(popup, btn, targetInput) {
    const btnRect = btn.getBoundingClientRect();
    const popupRect = popup.getBoundingClientRect();
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;

    // Start with default position (above the button)
    let top = btnRect.top - popupRect.height - 8;
    let left = btnRect.right - popupRect.width;

    // Check if there are important UI elements to avoid
    const uiElements = detectImportantUIElements();

    // Optimize initial position based on viewport
    // If button is too close to the top, position below
    if (top < 10) {
      top = btnRect.bottom + 8;
    }

    // Don't let popup go off-screen horizontally
    if (left < 10) {
      left = 10;
    } else if (left + popupRect.width > viewportWidth - 10) {
      left = viewportWidth - popupRect.width - 10;
    }

    // Check if popup would overlap with important UI elements
    const popupBox = {
      top,
      left,
      right: left + popupRect.width,
      bottom: top + popupRect.height,
    };

    // Try to avoid overlaps with important UI
    for (const element of uiElements) {
      if (boxesOverlap(popupBox, element)) {
        // Try positioning to different sides
        const positions = [
          { top: btnRect.bottom + 8, left }, // Bottom
          { top: btnRect.top - popupRect.height - 8, left }, // Top
          { top: btnRect.top, left: btnRect.right + 8 }, // Right
          { top: btnRect.top, left: btnRect.left - popupRect.width - 8 }, // Left
        ];

        // Find first position that doesn't overlap
        for (const pos of positions) {
          const testBox = {
            top: pos.top,
            left: pos.left,
            right: pos.left + popupRect.width,
            bottom: pos.top + popupRect.height,
          };

          if (
            !uiElements.some((el) => boxesOverlap(testBox, el)) &&
            isWithinViewport(testBox, viewportWidth, viewportHeight)
          ) {
            top = pos.top;
            left = pos.left;
            break;
          }
        }

        break;
      }
    }

    // Apply final position
    popup.style.position = "fixed";
    popup.style.left = `${left}px`;
    popup.style.top = `${top}px`;
    // Track the input element to enable applying styles later
    popup.targetInput = targetInput;
  }

  // Helper function to detect if a position is within viewport
  function isWithinViewport(box, viewportWidth, viewportHeight) {
    return (
      box.top >= 0 &&
      box.left >= 0 &&
      box.right <= viewportWidth &&
      box.bottom <= viewportHeight
    );
  }

  // Helper function to check if two boxes overlap
  function boxesOverlap(box1, box2) {
    return !(
      box1.right < box2.left ||
      box1.left > box2.right ||
      box1.bottom < box2.top ||
      box1.top > box2.bottom
    );
  }

  // Detect important UI elements to avoid overlapping
  function detectImportantUIElements() {
    const elements = [];
    const currentUrl = window.location.href;

    // Social media specific elements to avoid
    if (currentUrl.includes("twitter.com") || currentUrl.includes("x.com")) {
      // Twitter navigation bar and sidebar
      document
        .querySelectorAll('header[role="banner"], nav[role="navigation"]')
        .forEach((el) => {
          const rect = el.getBoundingClientRect();
          if (rect.width > 0 && rect.height > 0) {
            elements.push({
              top: rect.top,
              left: rect.left,
              right: rect.right,
              bottom: rect.bottom,
            });
          }
        });
    } else if (currentUrl.includes("facebook.com")) {
      // Facebook header and sidebars
      document
        .querySelectorAll('div[role="banner"], div[data-pagelet="LeftRail"]')
        .forEach((el) => {
          const rect = el.getBoundingClientRect();
          if (rect.width > 0 && rect.height > 0) {
            elements.push({
              top: rect.top,
              left: rect.left,
              right: rect.right,
              bottom: rect.bottom,
            });
          }
        });
    } else if (currentUrl.includes("linkedin.com")) {
      // LinkedIn header and sidebars
      document.querySelectorAll("header, aside").forEach((el) => {
        const rect = el.getBoundingClientRect();
        if (rect.width > 0 && rect.height > 0) {
          elements.push({
            top: rect.top,
            left: rect.left,
            right: rect.right,
            bottom: rect.bottom,
          });
        }
      });
    }

    // Generic fixed elements check - likely navigation or important UI
    document.querySelectorAll("*").forEach((el) => {
      if (
        window.getComputedStyle(el).position === "fixed" &&
        el.className &&
        !el.className.includes("unicode-")
      ) {
        const rect = el.getBoundingClientRect();
        if (rect.width > 100 && rect.height > 30) {
          // Minimum size to consider
          elements.push({
            top: rect.top,
            left: rect.left,
            right: rect.right,
            bottom: rect.bottom,
          });
        }
      }
    });

    return elements;
  }

  function isTextInputElement(element) {
    if (!element || !(element instanceof HTMLElement)) return false;

    // Exclude Quill clipboard helper
    if (
      element.classList.contains("ql-clipboard") &&
      element.getAttribute("tabindex") === "-1"
    ) {
      return false;
    } // Only allow editable elements, but exclude password fields
    const isStandardTextInput =
      (element instanceof HTMLInputElement && element.type === "text") ||
      element instanceof HTMLTextAreaElement; // Exclude password fields completely
    const isPasswordField =
      element instanceof HTMLInputElement && element.type === "password";

    // Additional password field detection by common attributes
    const hasPasswordAttributes =
      element
        .getAttribute("autocomplete")
        ?.toLowerCase()
        .includes("password") ||
      element.getAttribute("name")?.toLowerCase().includes("password") ||
      element.getAttribute("id")?.toLowerCase().includes("password") ||
      element.getAttribute("aria-label")?.toLowerCase().includes("password") ||
      element.getAttribute("placeholder")?.toLowerCase().includes("password");

    const isContenteditable =
      element.getAttribute("contenteditable") === "true";

    // Instagram-specific detection
    const isInstagramInput =
      (element
        .getAttribute("aria-label")
        ?.toLowerCase()
        .includes("add a comment") ||
        element
          .getAttribute("placeholder")
          ?.toLowerCase()
          .includes("add a comment")) &&
      element instanceof HTMLTextAreaElement;

    // Specific social media editors must be editable
    const isSocialMediaEditor =
      (element.classList.contains("public-DraftEditor-content") &&
        element.isContentEditable) ||
      (element.classList.contains("notranslate") &&
        element.isContentEditable) ||
      (element.getAttribute("role") === "textbox" &&
        element.isContentEditable) ||
      (element.getAttribute("data-testid") === "tweetTextarea_0" &&
        element.isContentEditable) ||
      (element.getAttribute("aria-label")?.toLowerCase().includes("tweet") &&
        element.isContentEditable) ||
      (element.classList.contains("ql-editor") && element.isContentEditable) ||
      (element.classList.contains("msg-form__contenteditable") &&
        element.isContentEditable) ||
      (element
        .getAttribute("data-placeholder")
        ?.toLowerCase()
        .includes("share") &&
        element.isContentEditable) ||
      (element.getAttribute("aria-label")?.toLowerCase().includes("comment") &&
        element.isContentEditable) ||
      (element.getAttribute("aria-label")?.toLowerCase().includes("post") &&
        element.isContentEditable) ||
      (element.getAttribute("placeholder")?.toLowerCase().includes("comment") &&
        element.isContentEditable);

    // Comprehensive search exclusion
    const isSearchInput =
      element.type === "search" ||
      element.getAttribute("role") === "searchbox" ||
      element.getAttribute("role") === "combobox" ||
      element.getAttribute("aria-label")?.toLowerCase().includes("search") ||
      element.getAttribute("placeholder")?.toLowerCase().includes("search") ||
      element.className?.toLowerCase().includes("search") ||
      element.id?.toLowerCase().includes("search") ||
      element.name?.toLowerCase() === "q" ||
      (element.getAttribute("autocomplete") === "off" &&
        element.name?.toLowerCase().includes("search"));

    // Additional check for tabindex="-1" for generic contenteditables,
    // but allow for specific known roles if they also have tabindex="-1"
    if (
      element.getAttribute("tabindex") === "-1" &&
      !(
        element instanceof HTMLInputElement ||
        element instanceof HTMLTextAreaElement
      ) &&
      !isSocialMediaEditor &&
      !isInstagramInput
    ) {
      // If it's a generic contenteditable with tabindex -1 and not a known specific editor type,
      // check its role. Allow if role is textbox, otherwise exclude.
      if (element.getAttribute("role") !== "textbox") {
        return false;
      }
    }
    const isEditable =
      isStandardTextInput ||
      isContenteditable ||
      isSocialMediaEditor ||
      isInstagramInput;

    return (
      isEditable && !isSearchInput && !isPasswordField && !hasPasswordAttributes
    );
  }

  function isElementSuitable(element) {
    if (!element || !(element instanceof HTMLElement)) return false;
    const rect = element.getBoundingClientRect();
    return rect.height >= MIN_INPUT_HEIGHT && rect.width >= MIN_INPUT_WIDTH;
  }

  function findDeepEditor(element) {
    if (!element || !(element instanceof HTMLElement)) {
      return null;
    }

    // Prioritize the element itself if it's a direct text input
    if (isTextInputElement(element)) {
      // Further checks for specific platforms if the initial element is too generic
      // For example, a generic div that is contenteditable might wrap a more specific editor
      if (
        element.matches('div[contenteditable="true"]') &&
        !element.className.includes("ql-editor") &&
        !element.className.includes("msg-form__contenteditable")
      ) {
        // Potentially too generic, try to find a more specific child or skip if it's already specific enough
      } else {
        return element;
      }
    }

    let editor = null;

    // Twitter specific
    editor =
      element.closest(
        '[data-testid="tweetTextarea_0"], [data-testid*="dmComposerSendButton"]'
      ) || element.querySelector('[data-testid="tweetTextarea_0"]');
    if (editor && isTextInputElement(editor)) return editor;

    // LinkedIn specific selectors
    const linkedInSelectors = [
      ".ql-editor", // Standard rich text editor
      ".msg-form__contenteditable", // Messaging input
      "textarea[id*='decorated-form-field']", // Comment boxes, post editor
      "div[aria-labelledby*='post-text-editor']", // Post editor
      "div[role='textbox'][aria-multiline='true']", // Generic multiline textbox
      "[data-artdeco-is-focused='true']", // Element that currently has focus styling from LinkedIn's JS
      "[aria-label*='message body']", // Message body input in new LinkedIn UI
      "[aria-label*='Write a post']",
      "[aria-label*='Start a post']",
      "[aria-placeholder*='What do you want to talk about?']",
      "[aria-label*='Share your thoughts']",
      "[aria-label*='Write an article']",
      "[aria-label*='Comment']",
      "textarea[name='message']", // Fallback for some message forms
    ];
    for (const selector of linkedInSelectors) {
      editor = element.closest(selector) || element.querySelector(selector);
      if (editor && isTextInputElement(editor)) return editor;
    }
    // Check if the element itself matches a LinkedIn selector (if it wasn't caught by isTextInputElement initially)
    for (const selector of linkedInSelectors) {
      if (element.matches(selector) && isTextInputElement(element))
        return element;
    }

    // Facebook specific
    editor =
      element.closest(
        '.notranslate[data-lexical-editor="true"], div[aria-label="Create a post"] div[role="textbox"]'
      ) ||
      element.querySelector(
        '.notranslate[data-lexical-editor="true"], div[aria-label="Create a post"] div[role="textbox"]'
      );
    if (editor && isTextInputElement(editor)) return editor;
    // Facebook comment specific (often a div with role textbox inside a form)
    const fbCommentBox = element.closest(
      'form div[role="textbox"][contenteditable="true"]'
    );
    if (fbCommentBox && isTextInputElement(fbCommentBox)) return fbCommentBox;

    // Instagram specific
    editor =
      element.closest(
        'textarea[aria-label*="comment" i], div[contenteditable="true"][role="textbox"][aria-label*="comment" i]'
      ) ||
      element.querySelector(
        'textarea[aria-label*="comment" i], div[contenteditable="true"][role="textbox"][aria-label*="comment" i]'
      );
    if (editor && isTextInputElement(editor)) return editor;
    editor = element.closest('textarea[placeholder*="Add a comment..." i]');
    if (editor && isTextInputElement(editor)) return editor;

    // Generic fallbacks
    if (element.matches("textarea") && isTextInputElement(element))
      return element;
    if (element.isContentEditable && isTextInputElement(element))
      return element;

    // If the element is a wrapper, search within it for a suitable editor
    const potentialEditor = element.querySelector(
      'textarea, [contenteditable="true"], [role="textbox"]'
    );
    if (potentialEditor && isTextInputElement(potentialEditor)) {
      return potentialEditor;
    }

    return null; // No suitable editor found
  }

  function setupInputElement(input) {
    if (
      !input ||
      !(input instanceof HTMLElement) ||
      input.hasAttribute("data-unicode-processed")
    ) {
      return;
    }

    const targetInput = findDeepEditor(input);
    if (!targetInput) {
      return;
    }

    // Assign a unique ID to the input field if it doesn't have one
    if (!targetInput.id) {
      targetInput.id = `unicode-input-${inputCounter++}`;
    }
    const inputId = targetInput.id;

    targetInput.setAttribute("data-unicode-processed", "true");
    if (!isElementSuitable(targetInput)) return;

    const container = targetInput.parentElement;
    if (!container) return;
    container.classList.add("relative"); // Ensure container is relative for button positioning

    const btn = createSuggestionButton(targetInput);
    btn.setAttribute("data-input-id", inputId); // Link button to input
    const popup = createSuggestionPopup();
    popup.setAttribute("data-input-id", inputId); // Link popup to input

    // Append button to the input's container for better relative positioning
    if (!container.contains(btn)) {
      container.appendChild(btn);
    }

    // Add dynamic positioning for the button
    function updateButtonPosition() {
      if (!targetInput || !btn) return;

      const inputRect = targetInput.getBoundingClientRect();
      const containerRect = container.getBoundingClientRect();

      // Position button relative to input field
      const rightOffset = containerRect.right - inputRect.right + 8;
      const bottomOffset = containerRect.bottom - inputRect.bottom + 5;

      btn.style.right = `${rightOffset}px`;
      btn.style.bottom = `${bottomOffset}px`;
    }

    // Initial positioning
    updateButtonPosition();

    // Update position on scroll and resize
    const updatePositionDebounced = debounce(updateButtonPosition, 100);
    window.addEventListener("scroll", updatePositionDebounced, true);
    window.addEventListener("resize", updatePositionDebounced);

    // Also update when input changes size (e.g., expanding text areas)
    const resizeObserver = new ResizeObserver(updatePositionDebounced);
    resizeObserver.observe(targetInput);

    // Append popup to body to avoid z-index issues with parent elements
    if (!document.body.contains(popup)) {
      document.body.appendChild(popup);
    }

    // Only show popup for focused editor
    function showButtonIfFocused() {
      // Hide all other buttons first
      document
        .querySelectorAll(".unicode-suggestion-btn")
        .forEach((otherBtn) => {
          if (otherBtn !== btn) {
            otherBtn.classList.add("hidden");
          }
        });
      // Hide all other popups
      document
        .querySelectorAll(".unicode-suggestion-popup")
        .forEach((otherPopup) => {
          if (otherPopup !== popup) {
            otherPopup.classList.add("hidden");
            otherPopup.classList.remove("scale-100", "opacity-100");
            otherPopup.classList.add("scale-95", "opacity-0");
          }
        });

      if (
        document.activeElement === targetInput &&
        isElementSuitable(targetInput)
      ) {
        const text = targetInput.value || targetInput.textContent || "";
        if (text.length >= 3) {
          // Only show if text length condition met
          btn.classList.remove("hidden");
        } else {
          btn.classList.add("hidden");
          popup.classList.add("hidden"); // Also hide popup if button is hidden
        }
        activeInput = targetInput; // Set as active input
      } else {
        btn.classList.add("hidden");
        // If the focus is not on the input, and not on the popup or button, hide the popup.
        if (
          activePopup === popup &&
          !popup.contains(document.activeElement) &&
          !btn.contains(document.activeElement)
        ) {
          popup.classList.add("hidden");
          popup.classList.remove("scale-100", "opacity-100");
          popup.classList.add("scale-95", "opacity-0");
          if (activePopup === popup) activePopup = null;
        }
        if (activeInput === targetInput) activeInput = null; // Clear active input if blurred
      }
    }

    targetInput.addEventListener("focus", () => {
      activeInput = targetInput;
      showButtonIfFocused();
    });
    targetInput.addEventListener("blur", () => {
      // Delay hiding to allow clicks on button/popup
      setTimeout(() => {
        // Check if focus is back on the input
        if (document.activeElement === targetInput) {
          return; // Don't hide if focus is back on input
        }

        // Check if focus is on elements within our popup or button
        const activeElement = document.activeElement;
        const isInPopup = activePopup && activePopup.contains(activeElement);
        const isInButton = btn.contains(activeElement);

        // Special check for tab buttons and other popup interactive elements
        const isTabButton =
          activeElement &&
          (activeElement.classList.contains("tab-button") ||
            activeElement.closest(".tab-button") ||
            activeElement.closest(".unicode-suggestion-popup"));

        if (!isInPopup && !isInButton && !isTabButton) {
          btn.classList.add("hidden");
          if (activePopup === popup) {
            popup.classList.add("hidden");
            popup.classList.remove("scale-100", "opacity-100");
            popup.classList.add("scale-95", "opacity-0");
            activePopup = null;
          }
        }

        if (
          activeInput === targetInput &&
          document.activeElement !== targetInput &&
          !isInPopup &&
          !isInButton &&
          !isTabButton
        ) {
          activeInput = null;
        }
      }, 150); // Slightly longer delay to account for click event processing
    });

    // Check if this is a Twitter editor for special handling
    const isTwitterEditor =
      targetInput.closest('[data-testid="tweetTextarea_0"]') !== null;

    let timeoutId, idleTimeoutId;

    const checkInput = () => {
      if (
        document.activeElement !== targetInput ||
        !isElementSuitable(targetInput)
      ) {
        // btn.classList.add("hidden"); // Button visibility handled by focus/blur
        // popup.classList.add("hidden"); // Popup visibility handled by focus/blur and button click
        return;
      }
      const text = targetInput.value || targetInput.textContent || "";

      // Update preview text from current content
      previewText = text.length > 0 ? text.substring(0, 3) : "Abc";

      // Update all preview spans with the current text
      updatePreviews(popup, previewText);

      if (text.length >= 3) {
        if (document.activeElement === targetInput)
          btn.classList.remove("hidden");
      } else {
        btn.classList.add("hidden");
        popup.classList.add("hidden");
        popup.classList.remove("scale-100", "opacity-100");
        popup.classList.add("scale-95", "opacity-0");
        if (activePopup === popup) activePopup = null;
      }
    };

    // Function to update all preview elements
    function updatePreviews(popupEl, text) {
      const previewElements = popupEl.querySelectorAll("[data-preview-style]");
      previewElements.forEach((el) => {
        const styleType = el.getAttribute("data-preview-style");
        el.textContent = convertText(text, styleType);
      });

      // Also update layer previews if present
      updateLayerPreviews(popupEl);
    }

    const resetIdleTimer = () => {
      clearTimeout(idleTimeoutId);
      if (activePopup === popup && !popup.classList.contains("hidden")) {
        // Check if this is the active popup
        idleTimeoutId = setTimeout(() => {
          popup.classList.add("hidden");
          popup.classList.remove("scale-100", "opacity-100");
          popup.classList.add("scale-95", "opacity-0");
          if (activePopup === popup) activePopup = null; // Clear if this was the one timed out
        }, IDLE_TIMEOUT);
      }
    };

    const hideActivePopup = () => {
      // This function is now part of showButtonIfFocused's logic for hiding other popups
      // We ensure only one popup can be active.
      if (activePopup && activePopup !== popup) {
        activePopup.classList.add("hidden");
        activePopup.classList.remove("scale-100", "opacity-100");
        activePopup.classList.add("scale-95", "opacity-0");
      }
      activePopup = popup; // Set the current popup as active
    };

    // Add these new functions to save and restore selection globally
    let savedSelection = null;

    // Save the current selection for later use
    function saveCurrentSelection() {
      const selection = window.getSelection();
      if (selection.rangeCount > 0) {
        savedSelection = {
          range: selection.getRangeAt(0).cloneRange(),
          text: selection.toString(),
        };
        return savedSelection.text;
      }
      return null;
    }

    // Restore a previously saved selection
    function restoreSelection(targetElement) {
      if (!savedSelection || !savedSelection.range) return false;

      try {
        const selection = window.getSelection();
        selection.removeAllRanges();
        selection.addRange(savedSelection.range);

        // For mobile, we need to ensure the element keeps focus
        if (targetElement) {
          setTimeout(() => {
            targetElement.focus();
          }, 10);
        }

        return true;
      } catch (e) {
        console.error("Error restoring selection:", e);
        return false;
      }
    }

    btn.addEventListener("click", (e) => {
      e.preventDefault();
      e.stopPropagation();

      // Ensure the associated input is focused
      if (document.activeElement !== targetInput) {
        targetInput.focus(); // This will trigger the focus listener which handles showing the button
      }

      // Hide any other popups and set this one as active
      hideActivePopup();

      // Save selection before showing popup
      const selectedText = saveCurrentSelection();

      popup.classList.toggle("hidden");
      if (!popup.classList.contains("hidden")) {
        // Use smart positioning
        smartPositionPopup(popup, btn, targetInput); // Pass btn as reference for positioning
        popup.classList.remove("scale-95", "opacity-0");
        popup.classList.add("scale-100", "opacity-100");

        // Use saved selection for preview if available
        if (selectedText && selectedText.length > 0) {
          previewText = selectedText.substring(0, 3);
        } else {
          const text = targetInput.value || targetInput.textContent || "";
          previewText = text.length > 0 ? text.substring(0, 3) : "Abc";
        }

        // Update all previews
        updatePreviews(popup, previewText);

        // Sync layer checkboxes with active layers
        popup.querySelectorAll(".style-layer-checkbox").forEach((checkbox) => {
          const layerStyle = checkbox.getAttribute("data-layer");
          checkbox.checked = activeLayers.includes(layerStyle);
        });

        resetIdleTimer();
      } else {
        popup.classList.remove("scale-100", "opacity-100");
        popup.classList.add("scale-95", "opacity-0");
      }
    });

    document.addEventListener("click", (e) => {
      if (
        activePopup === popup && // Only act if this is the active popup
        !popup.classList.contains("hidden") &&
        !popup.contains(e.target) &&
        !btn.contains(e.target) && // Check if the click was on this instance's button
        targetInput !== e.target // Also ensure click is not back on the input field itself
      ) {
        popup.classList.add("hidden");
        popup.classList.remove("scale-100", "opacity-100");
        popup.classList.add("scale-95", "opacity-0");
        activePopup = null; // Clear active popup
        clearTimeout(idleTimeoutId);
      }
    }); // Add hover preview effects and click handling
    popup
      .querySelectorAll(
        ".suggestion-options button, .currency-options button, .special-chars-options button"
      )
      .forEach((option) => {
        // Add hover effects with preview changes
        option.addEventListener("mouseenter", () => {
          const previewEl = option.querySelector(".unicode-preview");
          if (previewEl) {
            previewEl.classList.add("highlighted");
          }
        });
        option.addEventListener("mouseleave", () => {
          const previewEl = option.querySelector(".unicode-preview");
          if (previewEl) {
            previewEl.classList.remove("highlighted");
          }
        });

        // Prevent option buttons from taking focus to avoid blur on input
        option.addEventListener("mousedown", (e) => {
          e.preventDefault();
        });

        // Handle style/currency application
        option.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();

          const cmd = option.getAttribute("data-cmd");
          const currencySymbol = option.getAttribute("data-currency");
          const specialChar = option.getAttribute("data-special-char");

          if (cmd) {
            // Focus the input and restore selection before applying style
            targetInput.focus();
            restoreSelection(targetInput);

            if (isTwitterEditor) {
              // Special handling for Twitter
              const twitterEditor = targetInput.closest(
                '[data-testid="tweetTextarea_0"]'
              );
              if (twitterEditor) {
                // Use setTimeout to ensure focus is maintained
                setTimeout(() => {
                  applyTwitterStyle(twitterEditor, cmd);
                }, 10);
              }
            } else {
              applyStyle(targetInput, cmd);
            }
          } else if (currencySymbol) {
            // For character insertion, just focus and insert at current cursor position
            targetInput.focus();
            insertTextAtCursor(targetInput, currencySymbol);
          } else if (specialChar) {
            // For character insertion, just focus and insert at current cursor position
            targetInput.focus();
            insertTextAtCursor(targetInput, specialChar);
          } // Don't auto-hide popup after applying styles - let users apply multiple styles
          // Only hide for currency/special char insertions since those are one-time insertions
          if (currencySymbol || specialChar) {
            popup.classList.add("hidden");
            popup.classList.remove("scale-100", "opacity-100");
            popup.classList.add("scale-95", "opacity-0");
            if (activePopup === popup) activePopup = null;
          }

          // Ensure focus is maintained and selection is preserved
          setTimeout(() => {
            targetInput.focus();
          }, 20);

          // Clear saved selection after use
          savedSelection = null;
        });
      });

    targetInput.addEventListener("keydown", (e) => {
      STYLES.forEach((style) => {
        const key = style.shortcut[0].split("+")[1]; // Get the key part (B, I, S, etc.)
        if (
          key === e.key.toUpperCase() &&
          ((isMac() && e.metaKey) || (!isMac() && e.ctrlKey)) &&
          !e.altKey &&
          !e.shiftKey
        ) {
          // Prevent default browser behavior (especially for Ctrl+D which selects the address bar)
          e.preventDefault();
          e.stopPropagation();

          // Apply the style
          if (isTwitterEditor) {
            const twitterEditor = targetInput.closest(
              '[data-testid="tweetTextarea_0"]'
            );
            if (twitterEditor) {
              applyTwitterStyle(twitterEditor, style.cmd);
            }
          } else {
            applyStyle(targetInput, style.cmd);
          }
        }
      });
    });

    if (targetInput.tagName === "INPUT" || targetInput.tagName === "TEXTAREA") {
      targetInput.addEventListener("input", () => {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(checkInput, 800);
      });
    } else if (targetInput.getAttribute("contenteditable") === "true") {
      const observer = new MutationObserver(() => {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(checkInput, 800);
      });
      observer.observe(targetInput, {
        childList: true,
        subtree: true,
        characterData: true,
      });
      targetInput.addEventListener("input", () => {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(checkInput, 800);
      });
    }
    loadTheme((currentTheme) => {
      const themeToggle = popup.querySelector("#theme-toggle");
      const icon = themeToggle.querySelector(".theme-icon");

      // Set initial icon based on current theme
      icon.innerHTML =
        currentTheme === "light"
          ? `<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z"/>`
          : `<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z"/>`;

      themeToggle.addEventListener("click", () => {
        const newTheme = toggleTheme(currentTheme);
        currentTheme = newTheme; // Update local state
        icon.innerHTML =
          newTheme === "light"
            ? `<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z"/>`
            : `<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z"/>`;
      });
    });

    checkInput();
  } // Add function to insert text at cursor position
  function insertTextAtCursor(input, text) {
    if (!input) return;

    // Handle LinkedIn editor specifically
    if (
      window.location.hostname.includes("linkedin.com") &&
      (input.classList.contains("ql-editor") ||
        input.closest(".ql-editor") ||
        input.classList.contains("msg-form__contenteditable"))
    ) {
      insertTextAtLinkedInCursor(input, text);
      return;
    }

    // Handle standard input and textarea elements
    if (input.tagName === "INPUT" || input.tagName === "TEXTAREA") {
      const start = input.selectionStart;
      const end = input.selectionEnd;
      const currentValue = input.value;

      // Replace selected text or insert at cursor position
      input.value =
        currentValue.substring(0, start) + text + currentValue.substring(end);

      // Set cursor position after inserted text
      const newCursorPos = start + text.length;
      input.setSelectionRange(newCursorPos, newCursorPos);

      // Maintain focus and trigger events
      input.focus();
      input.dispatchEvent(new Event("input", { bubbles: true }));
      input.dispatchEvent(new Event("change", { bubbles: true }));
      return;
    }

    // Handle contentEditable elements
    if (
      input.isContentEditable ||
      input.getAttribute("contenteditable") === "true"
    ) {
      insertTextAtContentEditableCursor(input, text);
      return;
    }

    // Fallback for other elements
    console.warn("Unsupported element type for text insertion:", input);
  }
  // Helper function to insert text at cursor in Twitter editor
  function insertTextAtTwitterCursor(editor, text) {
    const contentDiv = editor.querySelector('[data-contents="true"]');
    if (!contentDiv) {
      console.error("Could not find Twitter content div");
      return;
    }

    // Focus the editor first
    editor.focus();
    contentDiv.focus();

    const selection = window.getSelection();
    if (selection.rangeCount > 0) {
      const range = selection.getRangeAt(0);

      // Ensure we're within the content div
      if (
        contentDiv.contains(range.commonAncestorContainer) ||
        contentDiv === range.commonAncestorContainer
      ) {
        // Delete any selected text first
        range.deleteContents();

        // Create and insert the new text node
        const textNode = document.createTextNode(text);
        range.insertNode(textNode);

        // Move cursor to end of inserted text
        range.setStartAfter(textNode);
        range.setEndAfter(textNode);
        range.collapse(true);

        // Update selection
        selection.removeAllRanges();
        selection.addRange(range);
      } else {
        // Selection outside content div, append to end
        contentDiv.appendChild(document.createTextNode(text));

        // Set cursor to end
        const newRange = document.createRange();
        newRange.selectNodeContents(contentDiv);
        newRange.collapse(false);
        selection.removeAllRanges();
        selection.addRange(newRange);
      }
    } else {
      // No selection, append to end and set cursor
      contentDiv.appendChild(document.createTextNode(text));

      // Set cursor to end
      const newRange = document.createRange();
      newRange.selectNodeContents(contentDiv);
      newRange.collapse(false);
      selection.removeAllRanges();
      selection.addRange(newRange);
    }

    // Trigger Twitter events
    triggerTwitterEvents(editor);

    // Ensure focus is maintained
    setTimeout(() => {
      editor.focus();
      contentDiv.focus();
    }, 10);
  }

  // Helper function to insert text at cursor in LinkedIn editor
  function insertTextAtLinkedInCursor(element, text) {
    // LinkedIn uses complex rich text editors, we need to be more careful

    // First, ensure the element has focus
    element.focus();

    // Try to use the browser's execCommand for better integration
    if (document.execCommand) {
      try {
        // Save the current selection
        const selection = window.getSelection();
        const range = selection.rangeCount > 0 ? selection.getRangeAt(0) : null;

        // Use execCommand to insert text - this works better with rich text editors
        const success = document.execCommand("insertText", false, text);

        if (success) {
          // Trigger input events that LinkedIn expects
          element.dispatchEvent(new Event("input", { bubbles: true }));
          element.dispatchEvent(new Event("keyup", { bubbles: true }));
          element.dispatchEvent(new Event("change", { bubbles: true }));

          // For Quill editors, trigger their specific events
          if (element.classList.contains("ql-editor")) {
            element.dispatchEvent(new Event("text-change", { bubbles: true }));
          }

          return;
        }
      } catch (e) {
        console.warn(
          "execCommand failed, falling back to manual insertion:",
          e
        );
      }
    }

    // Fallback to manual DOM manipulation
    const selection = window.getSelection();

    if (selection.rangeCount > 0) {
      const range = selection.getRangeAt(0);

      // Ensure we're within the target element
      if (
        element.contains(range.commonAncestorContainer) ||
        element === range.commonAncestorContainer
      ) {
        // Delete any selected text first
        range.deleteContents();

        // Create and insert the new text node
        const textNode = document.createTextNode(text);
        range.insertNode(textNode);

        // Move cursor to end of inserted text
        range.setStartAfter(textNode);
        range.setEndAfter(textNode);
        range.collapse(true);

        // Update selection
        selection.removeAllRanges();
        selection.addRange(range);
      } else {
        // Selection outside element, append to end
        element.appendChild(document.createTextNode(text));

        // Set cursor to end
        const newRange = document.createRange();
        newRange.selectNodeContents(element);
        newRange.collapse(false);
        selection.removeAllRanges();
        selection.addRange(newRange);
      }
    } else {
      // No selection, append to end
      element.appendChild(document.createTextNode(text));

      // Set cursor to end
      const newRange = document.createRange();
      newRange.selectNodeContents(element);
      newRange.collapse(false);
      selection.removeAllRanges();
      selection.addRange(newRange);
    }
    // Trigger LinkedIn-specific events
    element.dispatchEvent(new Event("input", { bubbles: true }));
    element.dispatchEvent(new Event("keyup", { bubbles: true }));
    element.dispatchEvent(new Event("change", { bubbles: true }));

    // Additional events for better LinkedIn integration
    element.dispatchEvent(
      new InputEvent("beforeinput", { bubbles: true, data: text })
    );
    element.dispatchEvent(
      new KeyboardEvent("keydown", { bubbles: true, key: "Unidentified" })
    );
    element.dispatchEvent(
      new KeyboardEvent("keypress", { bubbles: true, key: "Unidentified" })
    );

    // For Quill editors specifically
    if (element.classList.contains("ql-editor")) {
      element.dispatchEvent(new Event("text-change", { bubbles: true }));
      // Also try to trigger on the parent Quill container
      const quillContainer = element.closest(".ql-container");
      if (quillContainer) {
        quillContainer.dispatchEvent(
          new Event("text-change", { bubbles: true })
        );

        // Try to access Quill instance directly if available
        if (quillContainer.quill) {
          try {
            quillContainer.quill.focus();
            const length = quillContainer.quill.getLength();
            quillContainer.quill.insertText(length - 1, text);
          } catch (e) {
            console.warn("Could not access Quill instance directly:", e);
          }
        }
      }
    }

    // Ensure focus is maintained
    setTimeout(() => {
      element.focus();
    }, 10);
  }

  // Helper function to insert text at cursor in contentEditable elements
  function insertTextAtContentEditableCursor(element, text) {
    // First try to get current selection
    const selection = window.getSelection();

    if (selection.rangeCount > 0) {
      const range = selection.getRangeAt(0);

      // Check if selection is within our target element
      if (
        element.contains(range.commonAncestorContainer) ||
        element === range.commonAncestorContainer
      ) {
        // Delete any selected text first
        range.deleteContents();

        // Create and insert the new text node
        const textNode = document.createTextNode(text);
        range.insertNode(textNode);

        // Move cursor to end of inserted text
        range.setStartAfter(textNode);
        range.setEndAfter(textNode);
        range.collapse(true);

        // Update selection
        selection.removeAllRanges();
        selection.addRange(range);
      } else {
        // Selection is outside our element, focus element and insert at end
        element.focus();
        element.appendChild(document.createTextNode(text));

        // Set cursor to end
        const newRange = document.createRange();
        newRange.selectNodeContents(element);
        newRange.collapse(false);
        selection.removeAllRanges();
        selection.addRange(newRange);
      }
    } else {
      // No selection, focus element and insert at end
      element.focus();
      element.appendChild(document.createTextNode(text));

      // Set cursor to end
      const newRange = document.createRange();
      newRange.selectNodeContents(element);
      newRange.collapse(false);
      selection.removeAllRanges();
      selection.addRange(newRange);
    }

    // Trigger input event
    element.dispatchEvent(new Event("input", { bubbles: true }));

    // Ensure focus is maintained
    setTimeout(() => {
      element.focus();
    }, 10);
  }

  function setupDeepDetection() {
    const observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        if (mutation.addedNodes.length) {
          mutation.addedNodes.forEach((node) => {
            if (!(node instanceof HTMLElement)) return;

            node
              .querySelectorAll(
                "input, textarea, [contenteditable='true'], [role='textbox'], " +
                  ".ql-editor, .msg-form__contenteditable, .public-DraftEditor-content, " +
                  ".notranslate, [data-text='true'], [data-testid='tweetTextarea_0'], " +
                  "[aria-label*='tweet' i], [aria-label*='post' i], [aria-label*='comment' i], " +
                  "[placeholder*='comment' i]"
              )
              .forEach((element) => {
                if (isTextInputElement(element)) setupInputElement(element);
              });
          });
        }
      });
    });

    observer.observe(document.body, { childList: true, subtree: true });

    document
      .querySelectorAll(
        "input, textarea, [contenteditable='true'], [role='textbox'], " +
          ".ql-editor, .msg-form__contenteditable, .public-DraftEditor-content, " +
          ".notranslate, [data-text='true'], [data-testid='tweetTextarea_0'], " +
          "[aria-label*='tweet' i], [aria-label*='post' i], [aria-label*='comment' i], " +
          "[placeholder*='comment' i]"
      )
      .forEach((element) => {
        if (isTextInputElement(element)) setupInputElement(element);
      });
  }

  function checkSocialEditors() {
    document
      .querySelectorAll(
        "input, textarea, [contenteditable='true'], [role='textbox'], " +
          ".ql-editor, .msg-form__contenteditable, .public-DraftEditor-content, " +
          ".notranslate, [data-text='true'], [data-testid='tweetTextarea_0'], " +
          "[aria-label*='tweet' i], [aria-label*='post' i], [aria-label*='comment' i], " +
          "[placeholder*='comment' i]"
      )
      .forEach((element) => {
        if (isTextInputElement(element)) setupInputElement(element);
      });
  }

  setupDeepDetection();
  checkSocialEditors();
  setInterval(checkSocialEditors, 2000);
}

function start() {
    const DISABLED_SITES_STORAGE_KEY = "viblifyDisabledSites";

    chrome.storage.sync.get([DISABLED_SITES_STORAGE_KEY], (result) => {
        const disabledSites = result[DISABLED_SITES_STORAGE_KEY] || [];
        const currentHostname = window.location.hostname.replace(/^www\./, "");

        if (disabledSites.includes(currentHostname)) {
            console.log("Viblify: Extension disabled on this site.");
            return;
        }
        initUnicodeSuggestions();
    });
}

if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", start);
} else {
  start();
}
